/*
 * linux/drivers/char/omap-keypad.c
 *
 * OMAP Keypad Driver
 *
 * Copyright (C) 2003 Nokia Corporation
 * Written by Timo Ter�s <ext-timo.teras@nokia.com>
 * iPAQ h6300 key and joypad support added by Mika Laitio. (2005)
 *
 * Added support for H2 & H3 Keypad
 * Copyright (C) 2004 Texas Instruments
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/input.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/mutex.h>
#include <asm/arch/irqs.h>
#include <asm/arch/gpio.h>
#include <asm/arch/hardware.h>
#include <asm/arch/keypad.h>
#include <asm/arch/menelaus.h>
#include <asm/io.h>
#include <asm/errno.h>
#include <asm/mach-types.h>
#include <asm/arch/mux.h>

#undef NEW_BOARD_LEARNING_MODE
//#define NEW_BOARD_LEARNING_MODE 1

/*
 * Following 5 keypad events are not really sent to userspace. 
 * Instead if the good combination of them is sent, then that is send.
 * (up, right, down, left, enter)
 */
#define	_H6300_JOYPAD_UP_RIGHT		1	// 00001
#define _H6300_JOYPAD_DOWN_RIGHT	2	// 00010
#define _h6300_JOYPAD_DOWN_LEFT		4	// 00100
#define _h6300_JOYPAD_UP_LEFT		8	// 01000
#define _H6300_JOYPAD_KEY_OK		16	// 10000
#define _H6300_JOYPAD_REPORT_COLUMN	4

static void omap_kp_tasklet(unsigned long);
static void omap_kp_timer(unsigned long);

static unsigned char keypad_state[8];
static DEFINE_MUTEX(kp_enable_mutex);
static int kp_enable = 1;
static int kp_cur_group = -1;

static int prevJoypadKeycodePressEmulated;

struct omap_kp {
	struct input_dev *input;
	struct timer_list timer;
	int irq;
	unsigned int rows;
	unsigned int cols;
};

DECLARE_TASKLET_DISABLED(kp_tasklet, omap_kp_tasklet, 0);

static int *keymap;
static unsigned int *row_gpios;
static unsigned int *col_gpios;

#ifdef CONFIG_ARCH_OMAP2
static void set_col_gpio_val(struct omap_kp *omap_kp, u8 value)
{
	int col;
	for (col = 0; col < omap_kp->cols; col++) {
		if (value & (1 << col))
			omap_set_gpio_dataout(col_gpios[col], 1);
		else
			omap_set_gpio_dataout(col_gpios[col], 0);
	}
}

static u8 get_row_gpio_val(struct omap_kp *omap_kp)
{
	int row;
	u8 value = 0;

	for (row = 0; row < omap_kp->rows; row++) {
		if (omap_get_gpio_datain(row_gpios[row]))
			value |= (1 << row);
	}
	return value;
}
#else
#define		set_col_gpio_val(x, y)	do {} while (0)
#define		get_row_gpio_val(x)	0
#endif

static irqreturn_t omap_kp_interrupt(int irq, void *dev_id,
				     struct pt_regs *regs)
{
	struct omap_kp *omap_kp = dev_id;

	/* disable keyboard interrupt and schedule for handling */
	if (cpu_is_omap24xx()) {
		int i;
		for (i = 0; i < omap_kp->rows; i++)
			disable_irq(OMAP_GPIO_IRQ(row_gpios[i]));
	} else {
		/* disable keyboard interrupt and schedule for handling */
		omap_writew(1, OMAP_MPUIO_BASE + OMAP_MPUIO_KBD_MASKIT);
	}

	tasklet_schedule(&kp_tasklet);

	return IRQ_HANDLED;
}

static void omap_kp_timer(unsigned long data)
{
	tasklet_schedule(&kp_tasklet);
}

static void omap_kp_scan_keypad(struct omap_kp *omap_kp, unsigned char *state)
{
	int col = 0;

	/* read the keypad status */
	if (cpu_is_omap24xx()) {
		int i;
		for (i = 0; i < omap_kp->rows; i++)
			disable_irq(OMAP_GPIO_IRQ(row_gpios[i]));
	} else {
		/* disable keyboard interrupt and schedule for handling */
		omap_writew(1, OMAP_MPUIO_BASE + OMAP_MPUIO_KBD_MASKIT);
	}
	if (!cpu_is_omap24xx()) {
		/* read the keypad status */
		omap_writew(0xff, OMAP_MPUIO_BASE + OMAP_MPUIO_KBC);
		for (col = 0; col < omap_kp->cols; col++) {
			omap_writew(~(1 << col) & 0xff, OMAP_MPUIO_BASE + OMAP_MPUIO_KBC);

			if (machine_is_omap_osk() || machine_is_omap_h2() || machine_is_omap_h3() || machine_is_omap_h6300()) {
				udelay(9);
			} else {
				udelay(4);
			}
			state[col] = ~omap_readw(OMAP_MPUIO_BASE + OMAP_MPUIO_KBR_LATCH) & 0xff;
		}
		omap_writew(0x00, OMAP_MPUIO_BASE + OMAP_MPUIO_KBC);
		udelay(2);
	} else {
		/* read the keypad status */
		for (col = 0; col < omap_kp->cols; col++) {
			set_col_gpio_val(omap_kp, ~(1 << col));
			state[col] = ~(get_row_gpio_val(omap_kp)) & 0x3f;
		}
		set_col_gpio_val(omap_kp, 0);
	}
}

static inline int omap_kp_find_key(int col, int row)
{
	int i, key;

	key = KEY(col, row, 0);
	for (i = 0; keymap[i] != 0; i++)
		if ((keymap[i] & 0xff000000) == key)
			return keymap[i] & 0x00ffffff;
	return -1;
}

int is_key_down(unsigned char new_state[],
		int col,
		int row)
{
	return (new_state[col] & (1 << row)) ? 1 : 0;
}

static void omap_kp_tasklet(unsigned long data)
{
	struct omap_kp *omap_kp_data = (struct omap_kp *) data;
	unsigned char new_state[8], changed, key_down = 0;
	int col, row;
	int spurious = 0;
	int report_key, report_col, report_row, joypad_checked;	// h6300-joypad specific variables

	/* check for any changes */
	omap_kp_scan_keypad(omap_kp_data, new_state);

	/* check for changes and print those */
	joypad_checked	= 0;
	for (col = 0; col < omap_kp_data->cols; col++) {
		changed = new_state[col] ^ keypad_state[col];
		key_down |= new_state[col];
		if (changed == 0)
			continue;

		for (row = 0; row < omap_kp_data->rows; row++) {
			int key;
			if (!(changed & (1 << row)))
				continue;
#ifdef NEW_BOARD_LEARNING_MODE
			printk(KERN_INFO "omap-keypad: key %d-%d %s\n", col, row, (new_state[col] & (1 << row)) ? "pressed" : "released");
#else
			key = omap_kp_find_key(col, row);
			if (key < 0) {
				printk(KERN_WARNING "omap-keypad: Spurious key event %d-%d\n",
				       col, row);
				/* We scan again after a couple of seconds */
				spurious = 1;
				continue;
			}
 			if (machine_is_omap_h6300() && 
 			   ((col == 1) || (col == _H6300_JOYPAD_REPORT_COLUMN)))
 			{
 				if (col == _H6300_JOYPAD_REPORT_COLUMN)
 				{
 					continue;
 				}
 				if ((joypad_checked == 0) &&
 				    ((key == _H6300_JOYPAD_KEY_OK) ||
 				     (key == _h6300_JOYPAD_UP_LEFT) ||
 	    			     (key == _H6300_JOYPAD_UP_RIGHT) ||
 				     (key == _H6300_JOYPAD_DOWN_RIGHT) ||
 				     (key == _h6300_JOYPAD_DOWN_LEFT)))
 				{
 					if (is_key_down(new_state, col, row))
 					{
 						/*
 						 * only enter pressed
 						 * 1 0 0 _H6300_JOYPAD_KEY_OK 0 0
 						 * --> 100100 == 36
 						 */
 						 if (new_state[1] == 36)
 						 {
 						 	joypad_checked	= 1;
 							prevJoypadKeycodePressEmulated	= KEY_ENTER;
 							new_state[_H6300_JOYPAD_REPORT_COLUMN]	= 48;	//110000
 							report_key	= prevJoypadKeycodePressEmulated;
 							report_col	= _H6300_JOYPAD_REPORT_COLUMN;
 							report_row	= 4;
 							input_report_key(omap_kp_data->input,
 									report_key,
 							 		new_state[report_col] & (1 << report_row));
 						 }						
 						/*
 						 * enter, up_left and up_right sensors pressed.
 						 * 1 _H6300_JOYPAD_UP_RIGHT 0 _H6300_JOYPAD_KEY_OK 0 _h6300_JOYPAD_UP_LEFT
 						 * --> 110101 == 53
 						 * OR
 						 * 1 KEY_UP_RIGHT 0 0 0 _h6300_JOYPAD_UP_LEFT
 						 * --> 110001 == 42
 						 * --> move to up
 						 */
 						else if ((new_state[1] == 53) ||
 						         (new_state[1] == 49))
 						{
 							joypad_checked	= 1;
 							prevJoypadKeycodePressEmulated	= KEY_UP;
 							new_state[_H6300_JOYPAD_REPORT_COLUMN]	= 40;	//101000
 							report_key	= prevJoypadKeycodePressEmulated;
 							report_col	= _H6300_JOYPAD_REPORT_COLUMN;
 							report_row	= 3;
 							input_report_key(omap_kp_data->input,
 									report_key,
 							 		new_state[report_col] & (1 << report_row));
 						}
 						/*
 						 * enter, down_left and down_right sensors pressed
 						 * --> 101110 == 46
 						 * OR
 						 * down_left and down_right
 						 * -->101010 == 42
 						 * --> move to down
 						 */
 						else if ((new_state[1] == 46) ||
 							 (new_state[1] == 42))
 						{
 							joypad_checked	= 1;
 							prevJoypadKeycodePressEmulated	= KEY_DOWN;
 							new_state[_H6300_JOYPAD_REPORT_COLUMN]	= 34;	//100010
 							report_key	= prevJoypadKeycodePressEmulated;
 							report_col	= _H6300_JOYPAD_REPORT_COLUMN;
 							report_row	= 1;
 							input_report_key(omap_kp_data->input,
 									report_key,
 							 		new_state[report_col] & (1 << report_row));
 						}												
 						/*
 						 * enter, up_right and down_right sensors pressed
 						 * --> 111100 == 60
 						 * or
 						 * down_right and up_right
 						 * --> 111000 == 56
 						 * --> move to right
 						 */
 						else if ((new_state[1] == 60) ||
 							 (new_state[1] == 56))
 						{
 							joypad_checked	= 1;
 							prevJoypadKeycodePressEmulated	= KEY_RIGHT;
 							new_state[_H6300_JOYPAD_REPORT_COLUMN]	= 33;	//100001
 							report_key	= prevJoypadKeycodePressEmulated;
 							report_col	= _H6300_JOYPAD_REPORT_COLUMN;
 							report_row	= 0;
 							input_report_key(omap_kp_data->input,
 									report_key,
 							 		new_state[report_col] & (1 << report_row));
 						}
 						/*
 						 * enter, up_left and down_left sensors pressed
 						 * --> 100111 == 39
 						 * or up_left and down_left
 						 * --> 100011 == 35
 						 * --> move to left
 						 */
 						else if ((new_state[1] == 39) ||
 							 (new_state[1] == 35))
 						{
 							joypad_checked	= 1;
 							prevJoypadKeycodePressEmulated	= KEY_LEFT;
 							new_state[_H6300_JOYPAD_REPORT_COLUMN]	= 36;	//100100
 							report_key	= prevJoypadKeycodePressEmulated;
 							report_col	= _H6300_JOYPAD_REPORT_COLUMN;
 							report_row	= 2;
 							input_report_key(omap_kp_data->input,
 									report_key,
 							 		new_state[report_col] & (1 << report_row));
 						}
 						else
 						{
 							//printk("missed new_state = %d\n", new_state[1]);
 						}
 					}
 					else
 					{
 						if (prevJoypadKeycodePressEmulated != 0)
 						{
 							// report key up event
 							joypad_checked	= 1;
 							new_state[_H6300_JOYPAD_REPORT_COLUMN]	= 32;	//100000
 							report_key	= prevJoypadKeycodePressEmulated;
 							report_col	= _H6300_JOYPAD_REPORT_COLUMN;
 							switch(prevJoypadKeycodePressEmulated)
 							{
 								case KEY_RIGHT:
									report_row	= 0;
 									break;
 								case KEY_DOWN:
 									report_row	= 1;
 									break;
 								case KEY_LEFT:
 									report_row	= 2;
 									break;
 								case KEY_UP:
 									report_row	= 3;
 									break;
 								case KEY_ENTER:
 									report_row	= 4;
 									break;
 								default:
 									printk(KERN_WARNING "Unknown iPAQ h6300 column 1 key = %d released. This should newer happen!\n",
 										key);
 									report_row	= 0;
 							}
 							input_report_key(omap_kp_data->input,
 									report_key,
 							 		new_state[report_col] & (1 << report_row));
 							prevJoypadKeycodePressEmulated	= 0;
 						}
 					}
 				}
 			}
 			else
 			{
				if (!(kp_cur_group == (key & GROUP_MASK) ||
				      kp_cur_group == -1))
					continue;

				kp_cur_group = key & GROUP_MASK;
				input_report_key(omap_kp_data->input, key & ~GROUP_MASK,
						 !!(new_state[col] & (1 << row)));
 			}
#endif
		}
	}
	memcpy(keypad_state, new_state, sizeof(keypad_state));

	if (key_down) {
                int delay = HZ / 20;
		/* some key is pressed - keep irq disabled and use timer
		 * to poll the keypad */
		if (spurious)
			delay = 2 * HZ;
		mod_timer(&omap_kp_data->timer, jiffies + delay);
	} else {
		/* enable interrupts */
		if (cpu_is_omap24xx()) {
			int i;
			for (i = 0; i < omap_kp_data->rows; i++)
				enable_irq(OMAP_GPIO_IRQ(row_gpios[i]));
		} else {
			omap_writew(0, OMAP_MPUIO_BASE + OMAP_MPUIO_KBD_MASKIT);
			kp_cur_group = -1;
		}
 	}
}

static ssize_t omap_kp_enable_show(struct device *dev, 
				   struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%u\n", kp_enable);
}

static ssize_t omap_kp_enable_store(struct device *dev, struct device_attribute *attr,
				    const char *buf, size_t count)
{
	int state;

	if (sscanf(buf, "%u", &state) != 1)
		return -EINVAL;

	if ((state != 1) && (state != 0))
		return -EINVAL;

	mutex_lock(&kp_enable_mutex);
	if (state != kp_enable) {
		if (state)
			enable_irq(INT_KEYBOARD);
		else
			disable_irq(INT_KEYBOARD);
		kp_enable = state;
	}
	mutex_unlock(&kp_enable_mutex);

	return strnlen(buf, count);
}

static DEVICE_ATTR(enable, S_IRUGO | S_IWUSR, omap_kp_enable_show, omap_kp_enable_store);

#ifdef CONFIG_PM
static int omap_kp_suspend(struct platform_device *dev, pm_message_t state)
{
	/* Nothing yet */

	return 0;
}

static int omap_kp_resume(struct platform_device *dev)
{
	/* Nothing yet */

	return 0;
}
#else
#define omap_kp_suspend	NULL
#define omap_kp_resume	NULL
#endif

static int __init omap_kp_probe(struct platform_device *pdev)
{
	struct omap_kp *omap_kp;
	struct input_dev *input_dev;
	struct omap_kp_platform_data *pdata =  pdev->dev.platform_data;
	int i;

	if (!pdata->rows || !pdata->cols || !pdata->keymap) {
		printk(KERN_ERR "No rows, cols or keymap from pdata\n");
		return -EINVAL;
	}

	omap_kp = kzalloc(sizeof(struct omap_kp), GFP_KERNEL);
	input_dev = input_allocate_device();
	if (!omap_kp || !input_dev) {
		kfree(omap_kp);
		input_free_device(input_dev);
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, omap_kp);

	omap_kp->input = input_dev;

	/* Disable the interrupt for the MPUIO keyboard */
	if (!cpu_is_omap24xx())
		omap_writew(1, OMAP_MPUIO_BASE + OMAP_MPUIO_KBD_MASKIT);

	keymap = pdata->keymap;

	if (pdata->rep)
		set_bit(EV_REP, input_dev->evbit);

	if (pdata->row_gpios && pdata->col_gpios) {
		row_gpios = pdata->row_gpios;
		col_gpios = pdata->col_gpios;
	}

	omap_kp->rows = pdata->rows;
	omap_kp->cols = pdata->cols;

	if (cpu_is_omap24xx()) {
		/* Cols: outputs */
		for (i = 0; i < omap_kp->cols; i++) {
			if (omap_request_gpio(col_gpios[i]) < 0) {
				printk(KERN_ERR "Failed to request"
				       "GPIO%d for keypad\n",
				       col_gpios[i]);
				return -EINVAL;
			}
			omap_set_gpio_direction(col_gpios[i], 0);
		}
		/* Rows: inputs */
		for (i = 0; i < omap_kp->rows; i++) {
			if (omap_request_gpio(row_gpios[i]) < 0) {
				printk(KERN_ERR "Failed to request"
				       "GPIO%d for keypad\n",
				       row_gpios[i]);
				return -EINVAL;
			}
			omap_set_gpio_direction(row_gpios[i], 1);
		}
	}
	prevJoypadKeycodePressEmulated		= 0;

	init_timer(&omap_kp->timer);
	omap_kp->timer.function = omap_kp_timer;
	omap_kp->timer.data = (unsigned long)omap_kp;

	/* get the irq and init timer*/
	tasklet_enable(&kp_tasklet);
	kp_tasklet.data = (unsigned long) omap_kp;

	omap_kp->irq = platform_get_irq(pdev, 0);
	if (omap_kp->irq >= 0) {
		if (request_irq(omap_kp->irq, omap_kp_interrupt, 0,
				"omap-keypad", omap_kp) < 0)
			return -EINVAL;
	}

	device_create_file(&pdev->dev, &dev_attr_enable);

	/* setup input device */
	set_bit(EV_KEY, input_dev->evbit);
	for (i = 0; keymap[i] != 0; i++)
		set_bit(keymap[i] & KEY_MAX, input_dev->keybit);
	input_dev->name = "omap-keypad";
	input_dev->cdev.dev = &pdev->dev;
	input_dev->private = omap_kp;
	input_register_device(omap_kp->input);

	if (machine_is_omap_h2() || machine_is_omap_h3() ||
	    machine_is_omap_perseus2() || machine_is_omap_h6300()) {
		omap_writew(0xff, OMAP_MPUIO_BASE + OMAP_MPUIO_GPIO_DEBOUNCING);
	}
	/* scan current status and enable interrupt */
	omap_kp_scan_keypad(omap_kp, keypad_state);
	if (!cpu_is_omap24xx()) {
		omap_writew(0, OMAP_MPUIO_BASE + OMAP_MPUIO_KBD_MASKIT);
	} else {
		for (i = 0; i < omap_kp->rows; i++) {
			if (request_irq(OMAP_GPIO_IRQ(row_gpios[i]), omap_kp_interrupt,
					SA_TRIGGER_FALLING, "omap-keypad", omap_kp) < 0)
				return -EINVAL;
		}
	}

	return 0;
}

static int omap_kp_remove(struct platform_device *pdev)
{
	struct omap_kp *omap_kp = platform_get_drvdata(pdev);

	/* disable keypad interrupt handling */
	tasklet_disable(&kp_tasklet);
	if (cpu_is_omap24xx()) {
		int i;
		for (i = 0; i < omap_kp->cols; i++)
	    		omap_free_gpio(col_gpios[i]);
		for (i = 0; i < omap_kp->rows; i++) {
	    		omap_free_gpio(row_gpios[i]);
			free_irq(OMAP_GPIO_IRQ(row_gpios[i]), 0);
		}
	} else {
		omap_writew(1, OMAP_MPUIO_BASE + OMAP_MPUIO_KBD_MASKIT);
		free_irq(omap_kp->irq, 0);
	}

	del_timer_sync(&omap_kp->timer);

	/* unregister everything */
	input_unregister_device(omap_kp->input);

	kfree(omap_kp);

	return 0;
}

static struct platform_driver omap_kp_driver = {
	.probe		= omap_kp_probe,
	.remove		= omap_kp_remove,
	.suspend	= omap_kp_suspend,
	.resume		= omap_kp_resume,
	.driver		= {
		.name	= "omap-keypad",
	},
};

static int __devinit omap_kp_init(void)
{
	printk(KERN_INFO "OMAP Keypad Driver\n");
	return platform_driver_register(&omap_kp_driver);
}

static void __exit omap_kp_exit(void)
{
	platform_driver_unregister(&omap_kp_driver);
}

module_init(omap_kp_init);
module_exit(omap_kp_exit);

MODULE_AUTHOR("Timo Ter�s");
MODULE_DESCRIPTION("OMAP Keypad Driver");
MODULE_LICENSE("GPL");
