/* 
 * Bluetooth interface driver for TI BRF6100 on h6300
 * 
 * Copyright (C) 2005 Mika Laitio <lamikr@cc.jyu.fi>
 * Ideas taken from the brf6150 bt driver made by Todd Blumer for the pxa hx4700.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/platform_device.h>

#include <asm/hardware.h>
#include <asm/arch/gpio.h>

#include <asm/arch/h6300_uart_info.h>
#include "h6300_bt_led.h"

static void
h6300_bt_configure(struct uart_omap_port *up, int enable)
{
	printk(KERN_NOTICE "h6300_bt_brf6100.c h6300_bt_configure() started, enable = %d\n", enable);
	
	// printk( KERN_NOTICE "h6300 configure bluetooth: %d\n", enable );
	if (enable == 0) {
		omap_set_gpio_dataout(GPIO_N_BT_RST, 1);	// turn off gpio, note 1 == off for negative gpios
		mdelay(5);
		h6300_clear_led(INDEX_BT_LED);
	}
	else if (enable == 1) {
		omap_set_gpio_dataout(GPIO_N_BT_RST, 1);	// turn on gpio, note 0 == on for negative gpios
		mdelay(5);				
	}
	else if (enable == 2) {
		/*
		 * BRF6150's RTS goes low when firmware is ready
		 * so check for CTS=1 (nCTS=0 -> CTS=1). Typical 150ms
		 */
/*		
		int tries = 0; 
		do 
		{
			mdelay(10);
		} 
		while ((BTMSR & MSR_CTS) == 0 && tries++ < 50);
*/				
		h6300_set_led(INDEX_BT_LED, 16, 16);
	}
	printk(KERN_NOTICE "h6300_bt_brf6100.c h6300_bt_configure() done\n");
}

static void
h6300_bt_set_txrx(struct uart_omap_port *up, int txrx)
{
	printk(KERN_NOTICE "h6300_bt_brf6100.c h6300_bt_set_txrx(), txrx = %d done\n", txrx);
	/* do nothing */
}

static int
h6300_bt_get_txrx(struct uart_omap_port *up)
{
	printk(KERN_NOTICE "h6300_bt_brf6100.c h6300_bt_get_txrx() done\n");
	/* do nothing */
	return 0;
}

static int
h6300_bt_probe(struct platform_device *pdev)
{
	struct h6300_uart_funcs *funcs = (struct h6300_uart_funcs *)pdev->dev.platform_data;

	omap_request_gpio(GPIO_BT_PWR_EN);		// ask bt_power_en gpio, remember to release in remove_function
	omap_set_gpio_direction(GPIO_BT_PWR_EN, 1);	// set gpio direction to be output
	omap_set_gpio_dataout(GPIO_BT_PWR_EN, 1);	// turn on gpio

	mdelay(200);

	omap_request_gpio(GPIO_N_BT_RST);		// ask bt_reset gpio, remember to release in remove_function
	omap_set_gpio_direction(GPIO_N_BT_RST, 1);	// set gpio direction to be output
	omap_set_gpio_dataout(GPIO_N_BT_RST, 0);	// turn on gpio, note 0 == on for negative gpios
	
	/* configure bluetooth UART */
	//h6300_gpio_mode(GPIO_NR_H6300_BT_RXD_MD);
	//h6300_gpio_mode(GPIO_NR_H6300_BT_TXD_MD);
	//h6300_gpio_mode(GPIO_NR_H6300_BT_UART_CTS_MD);
	//h6300_gpio_mode(GPIO_NR_H6300_BT_UART_RTS_MD);

	funcs->configure	= h6300_bt_configure;
	funcs->set_txrx		= h6300_bt_set_txrx;
	funcs->get_txrx		= h6300_bt_get_txrx;

	/* Make sure the LED is off */
	h6300_clear_led(INDEX_BT_LED);
	
	printk(KERN_NOTICE "h6300_bt_brf6100.c h6300_bt_probe() done\n");	

	return 0;
}

static int
h6300_bt_remove(struct platform_device *pdev)
{
	struct h6300_uart_funcs *funcs = (struct h6300_uart_funcs *)pdev->dev.platform_data;
	
	printk(KERN_NOTICE "h6300_bt_brf6100.c h6300_bt_remove() started\n");	
	
	omap_free_gpio(GPIO_BT_PWR_EN);
	omap_free_gpio(GPIO_N_BT_RST);

	funcs->configure	= NULL;
	funcs->set_txrx		= NULL;
	funcs->get_txrx		= NULL;

	/* Make sure the LED is off */
	h6300_clear_led(INDEX_BT_LED);
	
	printk(KERN_NOTICE "h6300_bt_brf6100.c, h6300_bt_remove() done\n");

	return 0;
}

static struct platform_driver bt_driver = {
	.probe    = h6300_bt_probe,
	.remove   = h6300_bt_remove,
	.driver = {
		.name =	"h6300_bt",
	},
};

static int __init
h6300_bt_init(void)
{
	printk(KERN_NOTICE "h6300 Bluetooth Driver init()\n");
	return platform_driver_register(&bt_driver);
}

static void __exit
h6300_bt_exit(void)
{
	printk(KERN_NOTICE "h6300 Bluetooth Driver exit()\n");
	platform_driver_unregister(&bt_driver);
}

module_init(h6300_bt_init);
module_exit(h6300_bt_exit);

MODULE_AUTHOR("Mika Laitio, <lamikr@cc.jyu.fi>");
MODULE_DESCRIPTION("iPAQ h6300 BRF6100 Bluetooth driver.");
MODULE_LICENSE("GPL");

