/*
 * Modified from board-h6300.c
 *
 * Code for generic OMAP board. Should work on many OMAP systems where
 * the device drivers take care of all the necessary hardware initialization.
 * Do not put any board specific code to this file; create a new machine
 * type if you need custom low-level initializations.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/input.h>

#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>
#include <asm/mach/flash.h>
#include <asm/mach/map.h>

#include <asm/arch/gpio.h>

#include <asm/arch/tc.h>
#include <asm/arch/usb.h>
#include <asm/arch/keypad.h>
#include <asm/arch/common.h>
#include <asm/arch/mcbsp.h>
#include <asm/arch/omap-alsa.h>
#include <asm/arch/h6300_uart_info.h>

#define _h6300_KEY_CALENDAR	67 	// xmodmap 75 aka F9
#define _H6300_KEY_TELEPHONE	68 	// xmodmap 76 aka F10
#define _H6300_KEY_HOMEPAGE	87 	// xmodmap 87 aka Num_Lock
#define _H6300_KEY_MAIL		88 	// xmodmap 88 aka Scroll_Lock

/*
 * Following 5 keypad events are not really sent to userspace. 
 * Instead if the good combination of them is sent, then that is send.
 * (up, right, down, left, enter)
 */
#define	_H6300_JOYPAD_UP_RIGHT		1	// 00001
#define _H6300_JOYPAD_DOWN_RIGHT	2	// 00010
#define _h6300_JOYPAD_DOWN_LEFT		4	// 00100
#define _h6300_JOYPAD_UP_LEFT		8	// 01000
#define _H6300_JOYPAD_KEY_OK		16	// 10000

static int h6300_keymap[] = {
	KEY(2, 0, _h6300_KEY_CALENDAR),		// address button in the bottom left	
	KEY(2, 3, _H6300_KEY_TELEPHONE),	// start call button in the bottom
	KEY(3, 1, _H6300_KEY_HOMEPAGE),		// stop call button in the bottom
	KEY(3, 4, _H6300_KEY_MAIL),		// messaging button in the bottom right

	KEY(0, 0, KEY_VOLUMEUP),	// volume up button in the right side
	KEY(0, 1, KEY_VOLUMEDOWN),	// volume down button in the right side
	KEY(3, 2, KEY_RECORD),		// record button in the left side
	
	KEY(1, 0, _h6300_JOYPAD_UP_LEFT),	
	KEY(1, 1, _h6300_JOYPAD_DOWN_LEFT),	
	KEY(1, 2, _H6300_JOYPAD_KEY_OK),		
	KEY(1, 3, _H6300_JOYPAD_DOWN_RIGHT),
	KEY(1, 4, _H6300_JOYPAD_UP_RIGHT),	
	
	KEY(4, 0, KEY_RIGHT),
	KEY(4, 1, KEY_DOWN),
	KEY(4, 2, KEY_LEFT),		
	KEY(4, 3, KEY_UP),
	KEY(4, 4, KEY_ENTER),

	0
};

/*
 * Bluetooth - Relies on h6300_bt module,
 * so make the calls indirectly through pointers. Requires that the
 * h6300_bt bluetooth module be loaded before any attempt to use
 * bluetooth (obviously).
 */

static struct h6300_uart_funcs bt_funcs;
static struct h6300_uart_funcs gsm_funcs;

static void h6300_bt_configure(struct uart_omap_port *up, int enable)
{
	printk(KERN_NOTICE "board-h6300.c, h6300_bt_configure() started\n");
	if (bt_funcs.configure != NULL)
		bt_funcs.configure(up, enable);
	printk(KERN_NOTICE "board-h6300.c, h6300_bt_configure() done\n");
}

static void h6300_bt_set_txrx(struct uart_omap_port *up, int txrx)
{
	printk(KERN_NOTICE "board-h6300.c, h6300_bt_set_txrx() started\n");
	if (bt_funcs.set_txrx != NULL)
	{
		printk(KERN_NOTICE "board-h6300.c, h6300_bt_set_txrx(), bt_funcs.set_txrx != NULL\n");
		bt_funcs.set_txrx(up, txrx);
	}
	printk(KERN_NOTICE "board-h6300.c, h6300_bt_set_txrx() done\n");
}

static int h6300_bt_get_txrx(struct uart_omap_port *up)
{
	int	retVal;
	
	printk(KERN_NOTICE "board-h6300.c, h6300_bt_get_txrx() started\n");
	if (bt_funcs.get_txrx != NULL)
	{
		retVal	= bt_funcs.get_txrx(up);
		printk(KERN_NOTICE "board-h6300.c, h6300_bt_get_txrx(), bt_funcs.get_trx != null, done, retVal %d\n", retVal);
		return retVal;
	}
	else
	{
		printk(KERN_NOTICE "board-h6300.c, h6300_bt_get_txrx() done, returning 0\n");
		return 0;
	}
}

static struct platform_omap_serial_funcs h6300_omap_bt_funcs = {
	.configure	= h6300_bt_configure,
	.set_txrx	= h6300_bt_set_txrx,
	.get_txrx	= h6300_bt_get_txrx,
};

struct platform_device btuart_device = {
	.name	= "h6300_bt",
	.id	= 1,
};
EXPORT_SYMBOL(btuart_device);

static void h6300_gsm_configure(struct uart_omap_port *up, int enable)
{
	printk(KERN_NOTICE "board-h6300.c, h6300_gsm_configure() started\n");
	if (gsm_funcs.configure != NULL)
		gsm_funcs.configure(up, enable);
	printk(KERN_NOTICE "board-h6300.c, h6300_gsm_configure() done\n");
}

static void h6300_gsm_set_txrx(struct uart_omap_port *up, int txrx)
{
	printk(KERN_NOTICE "board-h6300.c, h6300_gsm_set_txrx() started\n");
	if (bt_funcs.set_txrx != NULL)
	{
		printk(KERN_NOTICE "board-h6300.c, h6300_gsm_set_txrx(), bt_funcs.set_txrx != NULL\n");
		gsm_funcs.set_txrx(up, txrx);
	}
	printk(KERN_NOTICE "board-h6300.c, h6300_gsm_set_txrx() done\n");
}

static int h6300_gsm_get_txrx(struct uart_omap_port *up)
{
	int	retVal;
	
	printk(KERN_NOTICE "board-h6300.c, h6300_gsm_get_txrx() started\n");
	if (bt_funcs.get_txrx != NULL)
	{
		retVal	= gsm_funcs.get_txrx(up);
		printk(KERN_NOTICE "board-h6300.c, h6300_gsm_get_txrx(), bt_funcs.get_trx != null, done, retVal %d\n", retVal);
		return retVal;
	}
	else
	{
		printk(KERN_NOTICE "board-h6300.c, h6300_gsm_get_txrx() done, returning 0\n");
		return 0;
	}
}

static struct platform_omap_serial_funcs h6300_omap_gsm_funcs = {
	.configure	= h6300_gsm_configure,
	.set_txrx	= h6300_gsm_set_txrx,
	.get_txrx	= h6300_gsm_get_txrx,
};

struct platform_device gsmuart_device = {
	.name	= "h6300_gsm",
	.id	= 1,
};
EXPORT_SYMBOL(gsmuart_device);

#if 0
static struct mtd_partition h6300_partitions[] = {
	/* bootloader (U-Boot, etc) in first sector */
	{
	      .name		= "bootloader",
	      .offset		= 0,
	      .size		= SZ_128K,
	      .mask_flags	= MTD_WRITEABLE, /* force read-only */
	},
	/* bootloader params in the next sector */
	{
	      .name		= "params",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= SZ_128K,
	      .mask_flags	= 0,
	},
	/* kernel */
	{
	      .name		= "kernel",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= SZ_2M,
	      .mask_flags	= 0
	},
	/* rest of flash1 is a file system */
	{
	      .name		= "rootfs",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= SZ_16M - SZ_2M - 2 * SZ_128K,
	      .mask_flags	= 0
	},
	/* file system */
	{
	      .name		= "filesystem",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= MTDPART_SIZ_FULL,
	      .mask_flags	= 0
	}
};

static struct flash_platform_data h6300_flash_data = {
	.map_name	= "cfi_probe",
	.width		= 2,
	.parts		= h6300_partitions,
	.nr_parts	= ARRAY_SIZE(h6300_partitions),
};

static struct resource h6300_flash_resource = {
	.start		= OMAP_CS0_PHYS,
	.end		= OMAP_CS0_PHYS + SZ_32M - 1,
	.flags		= IORESOURCE_MEM,
};

static struct platform_device h6300_flash_device = {
	.name	= "omapflash",
	.id		= 0,
	.dev	= {
		.platform_data	= &h6300_flash_data,
	},
static struct platform_device h6300_flash_device = {
	.name	= "omapflash",
	.id		= 0,
	.dev	= {
		.platform_data	= &h6300_flash_data,
	},
	.num_resources	= 1,
	.resource	= &h6300_flash_resource,
};
#endif

static struct resource h6300_kp_resources[] = {
	[0] = {
		.start	= INT_KEYBOARD,
		.end	= INT_KEYBOARD,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct omap_kp_platform_data h6300_kp_data = {
	.rows	= 8,
	.cols	= 8,
	.keymap = h6300_keymap,
	.rep	= 1,	// turns repeat bit on
};

static struct platform_device h6300_kp_device = {
	.name		= "omap-keypad",
	.id		= -1,
	.dev		= {
		.platform_data = &h6300_kp_data,
	},
	.num_resources	= ARRAY_SIZE(h6300_kp_resources),
	.resource	= h6300_kp_resources,
};

static struct resource h6300_wlan_resource[] = {
        [0] = {
                .start          = OMAP_CS1_PHYS,
                .end            = OMAP_CS1_PHYS + SZ_32M -1,
                .flags          = IORESOURCE_MEM,
        },

        [1] = {
                .start  = OMAP_GPIO_IRQ(11),
                .end    = OMAP_GPIO_IRQ(11),
                .flags  = IORESOURCE_IRQ,
        },
};

static struct platform_device h6300_wlan_device = {
        .name           = "tnetw1100b",
        .id             = 0,
        .num_resources  = 2,
        .resource       = h6300_wlan_resource,
};

static struct omap_mcbsp_reg_cfg mcbsp_regs = { 
	.spcr2	= 0x0000,
	.spcr1	= 0x0000,
	.rcr2	= 0x8041,
	.rcr1	= 0x0040,
	.xcr2	= 0x00a1,
	.xcr1	= 0x00a0,
	.srgr2	= 0xb000,
	.srgr1	= 0x0000,
	.pcr0	= 0x0081,
};

static struct omap_alsa_codec_config alsa_config = {
	.name                   = "iPAQ h6300 TSC2101",
	.mcbsp_regs_alsa        = &mcbsp_regs,
	.codec_configure_dev    = NULL, //tsc2101_configure,
	.codec_set_samplerate   = NULL, //tsc2101_set_samplerate,
	.codec_clock_setup      = NULL, //tsc2101_clock_setup,
	.codec_clock_on         = NULL, //tsc2101_clock_on,
	.codec_clock_off        = NULL, //tsc2101_clock_off,
	.get_default_samplerate = NULL, //tsc2101_get_default_samplerate,
};

static struct platform_device h6300_mcbsp1_device = {
	.name	= "omap_alsa_mcbsp",
	.id	= 1,
	.dev = {
		.platform_data	= &alsa_config,
	},
};

static struct platform_device h6300_lcd_device = {
	.name	= "lcd_h6300",
	.id	= -1,
};

static struct platform_device *h6300_devices[] __initdata = {
	&h6300_lcd_device,
	&btuart_device,
	&gsmuart_device,
	&h6300_kp_device,
	&h6300_mcbsp1_device,
	&h6300_wlan_device,
	//&h6300_flash_device,
};

static void __init h6300_init_irq(void)
{
	omap1_init_common_hw();
	omap_init_irq();
	omap_gpio_init();

	/* this is now done in the drivers/input/touschreen/omap/ts_hx.c*/
	//omap_request_gpio(2);
	//omap_set_gpio_direction(2, 0);
	//omap_set_gpio_dataout(2, 1);
}

/* assume no Mini-AB port */

static struct omap_usb_config h6300_usb_config __initdata = {
	.hmc_mode	= 0,
	.register_dev	= 1,
	.pins[0]	= 0,
};

static struct omap_lcd_config h6300_lcd_config __initdata = {
	.ctrl_name	= "internal",
};

static struct omap_mmc_config h6300_mmc_config __initdata = {
	.mmc [0] = {
		.enabled 	= 1,
		.wire4		= 1,
		.wp_pin		= OMAP_GPIO_IRQ(13),
		.power_pin	= -1, // tps65010 ?
		.switch_pin	= -1, // OMAP_MPUIO(1),	// = -1, // ARMIO2?
	},
};

static struct omap_uart_config h6300_uart_config __initdata = {
	.enabled_uarts = ((1 << 0) | (1 << 1) | (1 << 2)),
};

static struct omap_board_config_kernel h6300_config[] = {
	{ OMAP_TAG_USB, 	&h6300_usb_config },
	{ OMAP_TAG_MMC,		&h6300_mmc_config },
	{ OMAP_TAG_UART,	&h6300_uart_config },
	{ OMAP_TAG_LCD,		&h6300_lcd_config },
};

static void __init h6300_init(void)
{
	int ret;
	
	ret = platform_add_devices(h6300_devices, ARRAY_SIZE(h6300_devices));
	if (ret) 
	{
		printk(KERN_WARNING "Unable to add h6300 platform devices like bluetooth");
	}
	omap_board_config	= h6300_config;
	omap_board_config_size	= ARRAY_SIZE(h6300_config);
	omap_serial_init();
}

static void __init h6300_map_io(void)
{
	omap1_map_common_io();
	
	btuart_device.dev.platform_data		= &h6300_omap_bt_funcs;
	gsmuart_device.dev.platform_data	= &h6300_omap_gsm_funcs;
}

MACHINE_START(OMAP_H6300, "HP iPAQ h6300")
	/* MAINTAINER("Everett Coleman II <gcc80x86@fuzzyneural.net>") */
	.phys_io		= 0xfff00000,
	.io_pg_offst	= ((0xfef00000) >> 18) & 0xfffc,
	.boot_params	= 0x10000100,
	.map_io			= h6300_map_io,
	.init_irq		= h6300_init_irq,
	.init_machine	= h6300_init,
	.timer			= &omap_timer,
MACHINE_END
