/*
 * linux/arch/arm/mach-omap/board-generic.c
 *
 * Modified from board-h6300.c
 *
 * Code for generic OMAP board. Should work on many OMAP systems where
 * the device drivers take care of all the necessary hardware initialization.
 * Do not put any board specific code to this file; create a new machine
 * type if you need custom low-level initializations.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>

#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>
#include <asm/mach/flash.h>
#include <asm/mach/map.h>

#include <asm/arch/fpga.h>
#include <asm/arch/gpio.h>
#include <asm/arch/tc.h>
#include <asm/arch/usb.h>

#include "../common.h"

static int __initdata h6300_serial_ports[OMAP_MAX_NR_PORTS] = {1, 1, 1};

static struct mtd_partition h6300_partitions[] = {
	/* bootloader (U-Boot, etc) in first sector */
	{
	      .name		= "bootloader",
	      .offset		= 0,
	      .size		= SZ_128K,
	      .mask_flags	= MTD_WRITEABLE, /* force read-only */
	},
	/* bootloader params in the next sector */
	{
	      .name		= "params",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= SZ_128K,
	      .mask_flags	= 0,
	},
	/* kernel */
	{
	      .name		= "kernel",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= SZ_2M,
	      .mask_flags	= 0
	},
	/* rest of flash1 is a file system */
	{
	      .name		= "rootfs",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= SZ_16M - SZ_2M - 2 * SZ_128K,
	      .mask_flags	= 0
	},
	/* file system */
	{
	      .name		= "filesystem",
	      .offset		= MTDPART_OFS_APPEND,
	      .size		= MTDPART_SIZ_FULL,
	      .mask_flags	= 0
	}
};

static struct flash_platform_data h6300_flash_data = {
	.map_name	= "cfi_probe",
	.width		= 2,
	.parts		= h6300_partitions,
	.nr_parts	= ARRAY_SIZE(h6300_partitions),
};

static struct resource h6300_flash_resource = {
	.start		= OMAP_CS0_PHYS,
	.end		= OMAP_CS0_PHYS + SZ_32M - 1,
	.flags		= IORESOURCE_MEM,
};

static struct platform_device h6300_flash_device = {
	.name		= "omapflash",
	.id		= 0,
	.dev		= {
		.platform_data	= &h6300_flash_data,
	},
	.num_resources	= 1,
	.resource	= &h6300_flash_resource,
};

/* Only FPGA needs to be mapped here. All others are done with ioremap */
static struct map_desc h6300_io_desc[] __initdata = {
{ OMAP1510_FPGA_BASE, OMAP1510_FPGA_START, OMAP1510_FPGA_SIZE,
	MT_DEVICE },
};

static struct platform_device *h6300_devices[] __initdata = {
	&h6300_flash_device,
};

static void __init h6300_init_irq(void)
{
	omap_init_irq();
	omap_gpio_init();
	omap1510_fpga_init_irq();
}

/* assume no Mini-AB port */

static struct omap_usb_config h6300_usb_config __initdata = {
	.hmc_mode	      = 0,
	.register_dev	  = 1,
	.pins[0]        = 0,
};

static struct omap_board_config_kernel h6300_config[] = {
	{ OMAP_TAG_USB,           &h6300_usb_config },
};

static void __init h6300_init(void)
{
	platform_add_devices(h6300_devices, ARRAY_SIZE(h6300_devices));
	h6300_config[0].data = &h6300_usb_config;

	omap_board_config = h6300_config;
	omap_board_config_size = ARRAY_SIZE(h6300_config);
}

static void __init h6300_map_io(void)
{
	omap_map_common_io();

	iotable_init(h6300_io_desc, ARRAY_SIZE(h6300_io_desc));
	udelay(10);	/* Delay needed for FPGA */

	/* Dump the Innovator FPGA rev early - useful info for support. */
	printk(KERN_CRIT, "H6300 FPGA Rev %d.%d Board Rev %d\n",
			fpga_read(OMAP1510_FPGA_REV_HIGH),
			fpga_read(OMAP1510_FPGA_REV_LOW),
			fpga_read(OMAP1510_FPGA_BOARD_REV));

	omap_serial_init(h6300_serial_ports);
}

MACHINE_START(H6300, "HP IPaq H6300")
	MAINTAINER("Everett Coleman II <gcc80x86@fuzzyneural.net>")
	BOOT_MEM(0x10000000, 0xfff00000, 0xfef00000)
	BOOT_PARAMS(0x10000100)
	MAPIO(h6300_map_io)
	INITIRQ(h6300_init_irq)
	INIT_MACHINE(h6300_init)
	.timer		= &omap_timer,
MACHINE_END
