/******************************************************************************
 *  uboot_load                                                                *
 *  Copyright (C) 2005 Everett Coleman <gcc80x86@fuzzyneural.net>             *
 *  http://fuzzyneural.net                                                    *
 *                                                                            *
 *  This program is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by      *
 *  the Free Software Foundation; either version 2 of the License, or         *
 *  (at your option) any later version.                                       *
 *                                                                            *
 *  This program is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *  GNU General Public License for more details.                              *
 *                                                                            *
 *  You should have received a copy of the GNU General Public License         *
 *  along with this program; if not, write to the Free Software               *
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.*
 ******************************************************************************/
#include <stdint.h>
#include <stdio.h>
#include "uboot_load.h"
#ifdef WIN32
# include <WIN/windows.h>

extern BOOL VirtualCopy (LPVOID lpvDestMem, LPVOID lpvSrcMem, DWORD dwsizeInBytes, DWORD dwProtectFlag);
#endif /* WIN32 */

/* CODE BORROWED FROM HARET!!!! */
#define PHYS_CACHE_SIZE 0x10000
#define PHYS_CACHE_MASK (PHYS_CACHE_SIZE - 1)
// The amount of physical memory locations to cache
#define PHYS_CACHE_COUNT 10
// Cache several last mapped physical memory for better effectivity
static uint8_t *phys_mem [PHYS_CACHE_COUNT];
// The physical address (multiple of 32K, if 1 then slot is free)
static uint32_t phys_base [PHYS_CACHE_COUNT] = { 1, 1, 1, 1, 1, 1, 1, 1,  };

/* We allocate windows in virtual address space for physical memory
 * in 64K chunks, however we always ensure there are at least 32K ahead
 * the address user requested.
 */
uint8_t *memPhysMap (uint32_t paddr)
{
  // Address should be aligned
  paddr &= ~3;

  uint32_t base = paddr & ~(PHYS_CACHE_MASK >> 1);
  uint32_t offs = paddr & (PHYS_CACHE_MASK >> 1);

  int slot = -1;
	int i=0;
  for (i = 0; i < PHYS_CACHE_COUNT; i++)
    if ((phys_base [i] == 1)
     || (phys_base [i] == base))
    {
      slot = i;
      break;
    }

  // If there is no empty slot, make it
  if (slot == -1)
  {
    slot = PHYS_CACHE_COUNT - 1;
    // This can lock up -- dunno why :-(
    VirtualFree (phys_mem [slot], 0, MEM_RELEASE);
    phys_base [slot] = 1;
  }

  // Move the cache element hit to the front (LRU)
  if (slot)
  {
    uint32_t tmp1 = phys_base [slot];
    memmove (phys_base + 1, phys_base, slot * sizeof (uint32_t));
    phys_base [0] = tmp1;

    uint8_t *tmp2 = phys_mem [slot];
    memmove (phys_mem + 1, phys_mem, slot * sizeof (uint8_t *));
    phys_mem [0] = tmp2;
  }

  // If slot has not been allocated yet, allocate it
  if (phys_base [0] == 1)
  {
    phys_base [0] = base;
    phys_mem [0] = (uint8_t *)VirtualAlloc (NULL, PHYS_CACHE_SIZE,
                                          MEM_RESERVE, PAGE_NOACCESS);
    // Map requested physical memory to our virtual address hole
    if (!VirtualCopy ((void *)phys_mem [0], (void *)(base / 256),
                      PHYS_CACHE_SIZE, PAGE_READWRITE | PAGE_PHYSICAL | PAGE_NOCACHE))
    {
      VirtualFree (phys_mem [0], 0, MEM_RELEASE);
      phys_mem [0] = NULL;
    }
  }

  // In the case of failure ...
  if (!phys_mem [0])
  {
//    Output( L"returning NULL..." );
		fprintf (stderr, "slot error for %08X: %d\n", paddr, slot);
    phys_base [0] = 1;
    return NULL;
  }

  return phys_mem [0] + offs;
}

void memPhysReset ()
{
	int i=0;
  for (i = 0; i < PHYS_CACHE_COUNT; i++)
    if (phys_mem [i])
    {
      VirtualFree (phys_mem [i], 0, MEM_RELEASE);
      phys_mem [i] = NULL;
      phys_base [i] = 1;
    }
}
/* END: CODE BORROWED FROM HARET!!!! */

int
memory_dump (FILE *tx, uint32_t addr, uint32_t size, const char *filename) {
	FILE *fd=fopen (filename, "w");
	uint32_t offset=0, x=0;
#ifdef DEBUG_PACKAGE
    cmd_print (tx, "[debug] address=0x%08x size=%u file=%s\r\n", addr, size, filename);
#endif /* DEBUG_PACKAGE */

		while (offset < size) {
			uint32_t chunk=((size-offset) < PHYS_CACHE_SIZE) ? size - offset : PHYS_CACHE_SIZE;
			uint8_t *data=memPhysMap (addr + offset);
			if (!data && (x=1)) {
				cmd_print (tx, "[error] %s:memPhysMap: error\r\n", __FUNCTION__);
				break;
			}

			if (fwrite (data, 1, chunk, fd) != chunk) {
				cmd_print (tx, "[error] fwrite: %s\r\n", strerror (x=errno));
				break;
			}
			offset+=PHYS_CACHE_SIZE;
		} fclose (fd);
		return x;
} /* memory_dump */

static void
print_region (FILE *tx, int size, uint32_t address, uint8_t *data, const char *string) {
	if (string)
		cmd_print (tx, "\"%s\" ", string);
	switch (size) {
		case 64:
			cmd_print (tx, "0x%08X: %016lx\r\n", address, *((uint64_t *)data));
			break;
		case 32:
			cmd_print (tx, "0x%08X: %08x\r\n", address, *((uint32_t *)data));
			break;
		case 16:
			cmd_print (tx, "0x%08X: %04x\r\n", address, *((uint16_t *)data));
			break;
		default:
			cmd_print (tx, "0x%08X: %02x\r\n", address, *((uint8_t *)data));
			break;
	} /* switch (size) */
} /* print_region */

int
dump_region (FILE *tx, int size, uint32_t addr, uint32_t count, uint32_t wait) {
	int i=0;
	uint8_t *data=memPhysMap (addr);

	wait=(wait) ? wait : 100;
	size=(size) ? size : 32;
	count=(count) ? count : 1;

	if (!data) {
		cmd_print (tx, "[error] %s:memPhysMap: error\r\n", __FUNCTION__);
		return 1;
	}

	for (i=0; i < count; i++) {
		print_region (tx, size, addr, data, 0);
		if (count > 1)
			Sleep (wait);
	}

	return 0;
} /* dump_region */

/* Watchpoint Functions
 */
typedef struct {
	char *string;
	uint32_t address;
	uint8_t size;
	uint8_t prev[8];
	uint8_t *data;
} watchp_t;

#define WP_COUNT 50
static watchp_t watch_points[WP_COUNT]={{0, 0, 0, {0, }, 0}, };
static size_t   wp_size=0;

int
watch_region (FILE *tx, int size, const char *string, uint32_t address) {
	int slot=-1, i=0;
	if (!string) {
		cmd_print (tx, "[error] must supply an id\r\n");
		return 0;
	}
	for (i=0; i < WP_COUNT; i++) {
		if (watch_points[i].string == 0) {
			slot=i;
		} else if (!strcasecmp (watch_points[i].string, string)) {
			slot=i;
			free (watch_points[i].string);
			watch_points[i].string=0;
		}
	}
	if (slot == -1) {
		cmd_print (tx, "[error] no slots left for watch points, clear some points!\r\n");
		return 0;
	}
	if ((watch_points[slot].data=memPhysMap (address)) == 0) {
		cmd_print (tx, "[error] %s:memPhysMap: error\r\n", __FUNCTION__);
		return 1;
	}
	if ((watch_points[slot].string=strdup (string)) == 0) {
		cmd_print (tx, "[error] strdup: %s\r\n", strerror (errno));
		return 1;
	}

	watch_points[slot].size=size;
	watch_points[slot].address=address;
	print_region (tx, size, address, watch_points[slot].data, string);
	wp_size++;
	return 0;
} /* watch_region */

inline int
wp_count () {
	return wp_size;
} /* wp_count */

int
wp_clear (FILE *tx, const char *string) {
	int i=0;
	if (!wp_size||!string)
		return 0;
	for (i=0; i < WP_COUNT; i++)
		if (watch_points[i].string && !strcasecmp (watch_points[i].string, string)) {
			free (watch_points[i].string);
			watch_points[i].string=0;
			wp_size--;
			return 0;
		}
	cmd_print (tx, "[warning] watchpoint '%s' does not exist!\r\n", string);
	return 0;
} /* wp_clear */

void
wp_erase () {
	int i=0;
	for (i=0; i < WP_COUNT; i++)
		if (watch_points[i].string) {
			free (watch_points[i].string);
			watch_points[i].string=0;
		}
} /* wp_erase */

void
wp_check (FILE *tx) {
	int i=0, x=0;
	for (i=0; i < WP_COUNT; i++) {
		uint8_t foo[8]={0, };
		if (!watch_points[i].string)
			continue;
		if (!watch_points[i].data)
			continue;
		if (!watch_points[i].size)
			continue;

		if ((watch_points[i].data=memPhysMap (watch_points[i].address)) == 0) {
			cmd_print (tx, "[error] %s:memPhysMap: error\r\n", __FUNCTION__);
			return;
		} memcpy (foo, watch_points[i].data, watch_points[i].size/8);
		if (!memcmp (foo, watch_points[i].prev, watch_points[i].size/8))
			continue;
		if (!x++)
			cmd_print (tx, "\r\n");
		print_region (tx, watch_points[i].size, watch_points[i].address, foo, watch_points[i].string);
		memcpy (watch_points[i].prev, foo, watch_points[i].size/8);
	}
} /* wp_check */

int
wp_list (FILE *tx) {
	int i=0;
	cmd_print (tx, "Watch Points:\r\n");
	for (i=0; i < WP_COUNT; i++) {
		if (!watch_points[i].string)
			continue;
		cmd_print (tx, "  %02dbit %08x %s\r\n", watch_points[i].size, watch_points[i].address, watch_points[i].string);
	} return 0;
} /* wp_list */
