#include <linux/setup.h>
#include <string.h>
#include <bootstrap.h>
#include <interrupts.h>
#include <dma.h>


void dump_tags() {
	struct tag *tag = (struct tag *) phys_to_virt((uint32_t) bootstrap_info.tags);

	while(tag->hdr.tag != ATAG_NONE){
		switch(tag->hdr.tag) {
		case ATAG_CORE:
			printf("Page size %ld byte(s)\n",tag->u.core.pagesize);
			printf("root-dev: %04X\n",tag->u.core.rootdev);
			break;
		case ATAG_CMDLINE:
			printf("cmdline : %s\n",tag->u.cmdline.cmdline);
			break;
		case ATAG_MEM:
			printf("Memory : 0x%0X, %ld\n",
			       tag->u.mem.start,
			       tag->u.mem.size);
			break;
		case ATAG_INITRD:
		case ATAG_INITRD2:
			printf("Initrd: 0x%08X, %ld\n",
			       tag->u.initrd.start,
			       tag->u.initrd.size);
			break;
		default:
			printf("Unkown tag : %ld\n", tag->hdr.tag);
			break;
		}

		tag = tag_next(tag);
	}
}

static struct tag *lookup_tag(int type){
	struct tag *tag = (struct tag *) phys_to_virt((uint32_t) bootstrap_info.tags);
	
	while(tag->hdr.tag != ATAG_NONE){
		if(tag->hdr.tag == type){
			break;
		}
		tag = tag_next(tag);
	}

	return tag;
}

#if defined(USE_TAG_SETUP)
static void setup_tags(uint32_t start, uint32_t size, char *cmd) {
        struct tag *tag    = (struct tag *) phys_to_virt((uint32_t) bootstrap_info.tags);

        // Core tag
        tag->hdr.tag         = ATAG_CORE;
        tag->hdr.size        = tag_size(tag_core);
        tag->u.core.flags    = 0;
        tag->u.core.pagesize = 0x00001000;
        tag->u.core.rootdev  = 0x0000; // not used, use kernel cmdline for this
        tag = tag_next (tag);

        // now the cmdline tag
        tag->hdr.tag = ATAG_CMDLINE;

        // tag header, zero-terminated string and round size to 32-bit words
        tag->hdr.size = (sizeof (struct tag_header) + strlen (cmd) + 1 + 3) >> 2;
        strcpy (tag->u.cmdline.cmdline, cmd);
        tag = tag_next (tag);

        // now the mem32 tag
        tag->hdr.tag    = ATAG_MEM;
        tag->hdr.size   = tag_size (tag_mem32);
        tag->u.mem.start= PHYS_MEM_START;
        tag->u.mem.size = PHYS_MEM_SIZE;
        tag = tag_next(tag);


        /* and now the initrd tag */
        tag->hdr.tag            = ATAG_INITRD2;
        tag->hdr.size           = tag_size (tag_initrd);
        tag->u.initrd.start     = virt_to_phys(start);
        tag->u.initrd.size      = size;
        tag = tag_next (tag);

        // now the NULL tag
        tag->hdr.tag = ATAG_NONE;
        tag->hdr.size = 0;
}
#endif

void create_initrd_tag(uint32_t start, uint32_t size){
	struct tag *tag   = lookup_tag(ATAG_INITRD2);
	int	    found = (tag->hdr.tag == ATAG_INITRD2);
	
	/* and now the initrd tag */
        tag->hdr.tag            = ATAG_INITRD2;
        tag->hdr.size           = tag_size (tag_initrd);
        tag->u.initrd.start     = virt_to_phys(start);
        tag->u.initrd.size      = size;
        tag = tag_next (tag);

	if(!found){
        	// now the NULL tag
        	tag->hdr.tag = ATAG_NONE;
        	tag->hdr.size = 0;
	}
}

void linboot(uint32_t start, uint32_t size){
	// Create initrd tag
	create_initrd_tag(start, size);

	// Mask and clear pending
	// interrupts
	printf("Disable interrupts\n");
	irq_maskall();	
	
	// Disable ALL DMA channels
	printf("Disable DMA channels\n");
	dma_disable_all();

	// Lets try and boot linux
	printf("Boot linux...\n");
	bootlinux();
}
