#include <malloc.h>
#include <string.h>

block_t *__free = NULL;

// Low level function to allocate pages
extern void *__pg_alloc(uint32_t flag, size_t size);

void setup_heap(size_t size) {
	// Allocate free pages
	__free = (block_t *) __pg_alloc(0, ALIGN(size, PAGE_SIZE));

	__free->next = NULL;
	__free->size = ALIGN(size, PAGE_SIZE) - BLOCKSIZE;
}

static block_t *
__resize(size_t size)
{
	/* Round the size to 4K
	 * then try an allocate page
	 */
	size_t  aligned = ALIGN(size, PAGE_SIZE);
	block_t *block	= (block_t *) __pg_alloc(0, aligned);
	
	if(!block) {
		// Failed to allocate extra space
		return NULL;
	}

	block->size = aligned - BLOCKSIZE;
	block->next = NULL;

	/*
	 * Add the new block to
	 * the free list.
	 */	
	free((void *) ((uint8_t *) block + BLOCKSIZE));

	return block;
}

static block_t *
__split(block_t *block, size_t size)
{
	ssize_t sz = (block->size + BLOCKSIZE) - size;
	
	if(sz >= SMALLBLOCK){
		/*
		 * Ok the block is more then we need lets
		 * Split it to two blocks.
		 */
		block_t *newblock = (block_t *) BLOCK(block,size); /* point to the 
								    * new block
								    */
		
		newblock->next 	= block->next;		/* Add the new block to the list */
		newblock->size 	= sz - BLOCKSIZE;	/* set the size */ 
		block->size 	= size - BLOCKSIZE; 	/* set the size of the allocated block */
		
		return newblock;
	}

	return block->next;
}

static block_t *
__allocate(size_t size)
{
	block_t *free 	= 0;
	block_t *block 	= NULL;
	block_t *blk 	= NULL;

	size = ALIGN((size + sizeof(block_t)), BSIZE);	

	do {
		/*
		 * Allocating the required block:
		 * 1. Walk throw the free list and find the
		 * the right size.
		 * 2. Resize the page if the free list is empty or we can't find 
		 * the right size.
		 * 3. Quit the loop if you have the reqired space or return NULL
		 * if there is no more space left in the page.
		 */
		for(blk = block = __free; block; 
			blk = block ,block = block->next){
			if((block->size + BLOCKSIZE) >= size){
				break;
			}
		}

		if(!block) {
			if(__resize(size) == NULL){
				/* No more space */
				return NULL;
			}

			continue;
		}
	} while(!block);

	/*
	 * Try and split the block 
	 * if the size is big then
	 * what is required 
	 */
	free 		= __split(block,size);
	block->next  	= NULL;

	if(blk == block){
		/*
		 * It's the first block
		 * in the free list.
		 */
		__free = free;
	}else {
		/*
		 * It's not the first block
		 */
		blk->next = free;
	}

	return block;
}

void *
malloc(size_t size)
{
        block_t *block = __allocate(size);
        return (block)? (void *) ((uint8_t *) block + BLOCKSIZE) : NULL;
}

void
free(void *ptr)
{
	block_t *block 	= NULL;
	block_t *n 	= NULL;
	block_t	*blk 	= NULL;

	if(!ptr){
		return;
	}

	block 	= (block_t *) ((uint8_t *) ptr - BLOCKSIZE);

	if(!__free){
		/*
		 * We don't need to walk throw
		 * free list becuase it's empty
		 */
		__free = block;
		return;
	} 
	
	/* search for the right please in the list */
	for(blk = (block_t *) __free; blk ; blk = blk->next){
		if(blk->next == block || block == blk){
			/* We are trying t free
			 * unallocated block.
			 */
			return;
		}else if(!blk->next || blk->next > block){
			/*
			 * We found the right place
			 */
			break;
		}
	}

	/*
	 * Calculate the pointer for
	 * the next block in the list
	 */
	n = (blk->next)? blk->next : blk;

	if(block == BLOCK(blk,blk->size + BLOCKSIZE) &&
	   n == BLOCK(block,block->size + BLOCKSIZE)){
		/*
		 * Marge the block with the next
		 * and the previous blocks
		 */

		blk->next  = n->next;
		blk->size += (block->size + BLOCKSIZE) + (n->size + BLOCKSIZE);
		block 	   = blk;
	}else if(blk && (block == BLOCK(blk,blk->size + BLOCKSIZE))){
		/*
	 	 * check if we can marge with
	 	 * previous free block.
	 	 */
		blk->size 	= blk->size + block->size + BLOCKSIZE;
		block 		= blk;
	}else if(n && (BLOCK(block,block->size + BLOCKSIZE) == n)){
		/*
 	 	 * Check if we can marge with the
	  	 * next free block.
	 	 */
		block->size += n->size + BLOCKSIZE;
		block->next  = n->next;
		blk->next    = block;
	} else {
		/*
		 * We can't marge with any block
		 */
		block->next = (blk > block)? blk : blk->next;
		blk->next   = (blk > block)? blk->next : block;
	}

	/*
 	 * Is it the first block in 
	 * the list.
	 */
	if(!blk || 
	   (__free == blk && block < blk)) {
		__free = block;
	}
}


void __debug_malloc(){
	block_t *blk = __free;

	while(blk){
		printf("block [%08X, %d, %08X]\n", blk, blk->size, blk->next);
		blk = blk->next;
	}
}
