#include "fat_fs.h"

void *fat_get_sb(harddisk_t *hdisk) {
	uint8_t	      	  *blk	   = (uint8_t *) malloc(get_blocksize());
	fat_boot_sector_t *bs 	   = NULL;
	volume_info_t 	  *vol     = NULL;
	fat_sb_t      	  *sb	   = NULL;

	/*
	 * read boot sector
	 */
	if(disk_bread(hdisk->start, 1, blk)){
		printf("Failed to read supper block\n");
		free(blk);
		return NULL;
	}

	/* Check if this is valid dos partition */
	if(blk[0x1fe] != 0x55 && blk[0x1ff] != 0xAA){
		printf("Invalid supper block magic %02X-%02X\n",
		       blk[0x1fe],blk[0x1ff]);
		free(blk);
		return NULL;
	}

	/* Get pointer to boot record. */
	bs  = (fat_boot_sector_t *) blk;

	/* Allocate space for the supper block */
	sb = (fat_sb_t *) malloc(sizeof(fat_sb_t));

	if(!bs->fat_length){
		// Volume information is in the end 
		// of boot record, it's FAT32
		// file system
		vol = (volume_info_t *) &blk[64];
		sb->fat_bits = 32;
	}else {
		vol = (volume_info_t *) &(bs->fat32_length);
	}

	if(sb->fat_bits == 32) {
		if(strncmp(FAT32_SIGN, vol->fs_type,8)){
			printf("Invalid filesystem %.*s\n",8,vol->fs_type);
			goto err;
		}

		sb->fat_length = bs->fat32_length;
	}else {
		sb->fat_length = bs->fat_length;
		sb->fat_bits  = 16;

		if(strncmp(FAT16_SIGN, vol->fs_type,8)){
			if(strncmp(FAT12_SIGN, vol->fs_type,8)){
				printf("Invalid filesystem %.*s\n",
				       8,vol->fs_type);
				goto err;
			}

			sb->fat_bits = 12;
		}
	}

	sb->sector_size  = bs->sector_size;
	sb->fat_start  	 = bs->reserved + hdisk->start;
	sb->root_start 	 = sb->fat_start + sb->fat_length * bs->fats;
	sb->cluster_size = bs->cluster_size;
	
	if(sb->fat_bits == 32) {
		sb->root_size  = sb->cluster_size;
		sb->data_start = sb->root_start - (sb->cluster_size * 2);
	}else {
		sb->root_size  = ((bs->dir_entries[1] * (int) 256 + 
				   bs->dir_entries[0]) * sizeof (dir_entry_t)) 
					/ bs->sector_size;
		sb->data_start = sb->root_start + sb->root_size - 
					(sb->cluster_size * 2);
	}

	// Lets try and cache the 
	// first block of FAT
	sb->fat      = (uint8_t *) malloc(sb->sector_size);
	sb->entries  = (sb->sector_size * 8) / sb->fat_bits;
	sb->first    = 0;
	sb->last     = sb->entries;

	/* Lets try and cache
	 * root directory.
	 */
	sb->root = (uint8_t *) malloc(sb->root_size * sb->sector_size);

	if(disk_bread(sb->root_start, sb->root_size, sb->root)){
                printf("Failed to read root directory\n");
		free(sb->root);
                goto err;
        }

	// Create data cache.
	sb->data = (uint8_t *) malloc(sb->cluster_size * sb->sector_size);
	

#if defined(HAVE_FS_DEBUG)
	printf("Start	   : %d\n", bs->root_cluster);
	printf("Reserved   : %d\n", bs->reserved);
	printf("FAT bits   : %d\n", sb->fat_bits);
	printf("FAT length : %d\n", sb->fat_length);
	printf("FAT start  : %d\n", sb->fat_start);
	printf("Root start : %d\n", sb->root_start);
	printf("Root size  : %d\n", sb->root_size);
	printf("Cluster size: %d\n", sb->cluster_size);
	printf("Data start  : %d\n", sb->data_start);
#endif

	free(blk);
	return sb;
err:
	free(blk);
	free(sb);

	return NULL;
}

filesystem_ops_t fat_ops;

filesystem_ops_t *__setup_fat() {
	fat_ops.get_sb = fat_get_sb;
	fat_ops.lookup = fat_lookup;
	fat_ops.read   = fat_read;
	
	return &fat_ops;
}
