/*
 * Copyright (c) 2005 Texas Instruments, Inc.
 *    Ported by SDG Systems, LLC
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation;
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE FOR ANY
 *  CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES 
 *  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
 *  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF 
 *  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS, 
 *  COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS 
 *  SOFTWARE IS DISCLAIMED.
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include "ti_bts.h"

#ifndef MAKEWORD
#define MAKEWORD(a, b)      ((unsigned short)(((unsigned char)(a)) | ((unsigned short)((unsigned char)(b))) << 8))
#endif

#define TI_MANUFACTURER_ID  13

/*
 * Common Init Script specific
 */
const char *
cis_create_filename(const unsigned char* cmdparms)
{
    static char bts_file[50];

    /* Check for TI's id */
    unsigned short manfid = MAKEWORD(cmdparms[8], cmdparms[9]);

    if (TI_MANUFACTURER_ID == manfid) {
        unsigned short version = MAKEWORD(cmdparms[10], cmdparms[11]);
        
        unsigned short chip =  (version & 0x7C00) >> 10;
        unsigned short min_ver = (version & 0x007F);
        unsigned short maj_ver = (version & 0x0380) >> 7;

        if (0 != (version & 0x8000)) {
            maj_ver |= 0x0008;
        }
        
        sprintf( bts_file, "TIInit_%d.%d.%d.bts", 
            (int)chip, (int)maj_ver, (int)min_ver);

        return &bts_file[0];
    }
    return NULL;
}

typedef struct
{
    unsigned int magic;
    unsigned int version;
    unsigned char future[24];
} cheader_t;


/* The value 0x42535442 stands for (in ASCII) BTSB */
/* which is Bluetooth Script Binary */
#define FILE_HEADER_MAGIC   0x42535442


bts_t *
bts_load_script(const char* fname, unsigned int* version)
{
    bts_t* bts = NULL;
    FILE* fp = fopen(fname, "rb");

    if (NULL != fp) {
        /* Read header */
        cheader_t header;

        /* Read header */
        if (1 == fread(&header, sizeof(header), 1, fp)) {
            /* Check magic number for correctness */
            if (header.magic == FILE_HEADER_MAGIC) {
                /* If user wants the version number */
                if (NULL != version) {
                    *version = header.version;
                }
                bts = (bts_t*)fp;
            }
        }
        /* If failed reading the file, close it */
        if (NULL == bts) {
            fclose(fp);
        }
    }
    return bts;
}

unsigned int
bts_next_action(const bts_t* bts_fp, unsigned char* action_buf,
    unsigned int nMaxSize, unsigned short* ptype)
{
    unsigned int bytes = 0;
    FILE* fp = (FILE*)bts_fp;
    unsigned char action_hdr[4];

    if (bts_fp == NULL)
        return 0;

    /* Each Action has the following: */
    /* UINT16 type of this action */
    /* UINT16 size of rest */
    /* BYTE[] action buffer (for HCI, includes the type byte e.g. 1 for hci command) */

    if (1 == fread(&action_hdr[0], sizeof(action_hdr), 1, fp)) {
        unsigned short type = *(unsigned short*)&action_hdr[0];
        unsigned short size = *(unsigned short*)&action_hdr[2];

        if (size <= nMaxSize) {
            int nread = fread(action_buf, sizeof(action_buf[0]), size, fp);

            if (nread == size) {
                *ptype = type;
                bytes = (unsigned int)size;
            }
        }
    }

    return bytes;
}

void
bts_unload_script(bts_t* bts_fp)
{
    FILE* fp = (FILE*)bts_fp;

    if (NULL != fp) {
        fclose(fp);
    }
}


